Attribute VB_Name = "PartsListToCSV"
'       This is a part of the source code for Pro/DESKTOP.
'       Copyright (C) 1999-2001 Parametric Technology Corporation.
'       All rights reserved.
'
'       File:PartsListToCSV.bas

'       This utility creates a comma separated value (.CSV) file for each parts list
'       table on the drawing. The drawing should contain at least one parts list table.
'       For each parts list table, the script prompts for a .CSV file, and writes the
'       data from the parts list to it. A suitable default file name is chosen.
'
'       An alternative interface is coded with "#If False" that allows a single
'       selected table to be output.
'
Option Explicit

Private app As ProDESKTOP
Private uc1 As userCommand
Private ext As IProDExtensibility

Private Const sep = ","

'localisation strings
Private Const notDrawing = 601
Private Const noPartsLists = 602

Public Sub OnStartUp()
    Set ext = GetApp
    Set uc1 = ext.AddUserCommand(barDrawing, menuDrawingTable, -1, "Write as CSV file", "WriteToCSV", "PartsListToCSV.WritePartsListToCSV")
#If False Then  ' process the selected parts list
    uc1.SetPrompt "Writes the selected parts list table to a Comma Seperated Value file"
#Else   ' process all parts lists on the drawing
    uc1.SetPrompt "Writes each parts list table on the drawing to a Comma Seperated Value file"
#End If
End Sub

Public Sub OnCloseDown()
    Set ext = GetApp
    Set uc1 = ext.GetUserCommand(barDrawing, menuDrawingTable, "Write as CSV file")
    ext.RemoveUserCommand uc1
End Sub

Sub WritePartsListToCSV()
    Set app = GetApp

    On Error Resume Next
    Dim doc As DrawingDocument
    Set doc = app.GetActiveDoc
    If doc Is Nothing Then
        MsgBox GetResourceString(notDrawing), vbInformation
        Exit Sub
    End If
    
    On Error GoTo 0

#If False Then  ' process the selected parts list
    
    On Error Resume Next
    Dim table As aPartsListTable

    Set table = doc.GetSingleSelection("PartsListTable")
    If table Is Nothing Then
        MsgBox "No Parts List Table selected", vbInformation
        Exit Sub
    End If
    On Error GoTo 0
    
    Dim fileName As String
    fileName = FileNameForPartsList(table)
    If fileName = "" Then Exit Sub
    If InStr(1, fileName, ".") = 0 Then fileName = fileName + ".csv"

    ProcessPartsList table, fileName

#Else   ' process all parts lists on the drawing
    
    Dim partsLists As ObjectSet
    Set partsLists = GetPartsListTables(doc.GetDrawing)

    If partsLists.GetCount = 0 Then
        MsgBox GetResourceString(noPartsLists), vbInformation
        Exit Sub
    End If

    Dim it As iterator
    Set it = app.GetClass("it").CreateAObjectIt(partsLists)

    it.start
    Do While it.IsActive
        Dim table As aPartsListTable
        Set table = it.Current
        
        Dim fileName As String
        fileName = FileNameForPartsList(table)
        If fileName <> "" Then
            If InStr(1, fileName, ".") = 0 Then fileName = fileName + ".csv"
            ProcessPartsList table, fileName
        End If
        it.Next
    Loop
#End If

End Sub

Private Function GetPartsListTables(dwg As aDrawing) As ObjectSet

    Dim sheetLists As ObjectSet
    Set sheetLists = dwg.GetSheets
    
    Dim sheetSet As iterator
    Set sheetSet = app.GetClass("it").CreateAObjectIt(sheetLists)
    
    Dim partsLists As ObjectSet
    Set partsLists = app.GetClass("ObjectSet").CreateAObjectSet()
    
    sheetSet.start
    Do While sheetSet.IsActive
        
        Dim contents As ObjectSet
        Set contents = sheetSet.Current.GetTables
            
        Dim it As iterator
        Set it = app.GetClass("it").CreateAObjectIt(contents)
        
        it.start
        Do While it.IsActive
            If it.Current.IsA("PartsListTable") Then
                partsLists.AddMember it.Current
            End If
            it.Next
        Loop
    
    sheetSet.Next
    Loop
    
    Set GetPartsListTables = partsLists
End Function

Private Function FileNameForPartsList(table As aPartsListTable) As String
    ' This prompts for the name of the CSV file using the name of the assembly in the same
    ' folder as the drawing, if it has been saved.

    Dim folder As String
    folder = table.GetFile.GetName
    If folder <> "" Then  ' drawing has been saved to disk
        folder = Left(folder, InStrRev(folder, "\"))
    Else
        folder = CurDir & "\"
    End If

    Dim assyName As String, assyFile As aFile
    Set assyFile = table.GetPartsList.GetDesign.GetFile
    assyName = assyFile.GetName

    If assyName <> "" Then  ' assembly has been saved to disk
        assyName = Right(assyName, Len(assyName) - InStrRev(assyName, "\"))
        assyName = Left(assyName, InStrRev(assyName, ".") - 1)
    Else
        assyName = assyFile.GetDocument.GetTitle    ' assembly must have a document
    End If

    FileNameForPartsList = InputBox("Enter pathname of CSV file", "Parts List for " & assyName, folder & assyName & ".csv")
End Function

Private Sub ProcessPartsList(table As aPartsListTable, fileName As String)
    Dim fn As Integer
    fn = FreeFile

    On Error GoTo CantOpenFile
    Open fileName For Output As #fn
    On Error GoTo 0

    Dim nRows As Integer, nNodes As Integer, node As Integer, row As Integer
    nRows = table.GetRowCount
    nNodes = table.GetNodeCount

    Dim column As aTableColumn, cell As aTableCell, group As aCalloutGroup

    Dim rowStr As String, valueStr As String, cellStr As String

    ' output table title
    rowStr = ""
    valueStr = table.GetTitle(0)
    AddValue rowStr, valueStr
    Print #fn, rowStr

    ' output column headings
    rowStr = ""
    For node = 1 To nNodes - 1
        Set column = table.GetColumn(node)
        If Not column Is Nothing Then
            valueStr = table.GetTitle(node)
#If False Then  ' diagnostics
            Debug.Print "Column:"; node, "Title:"; valueStr, "Show:"; TableColumnType(table, column)
#End If
            AddValue rowStr, valueStr
        End If
    Next node

    Print #fn, rowStr

    ' output rows
    For row = 0 To nRows - 1
        rowStr = ""
        
        For node = 1 To nNodes - 1
            Set column = table.GetColumn(node)
            If Not column Is Nothing Then
                Set cell = column.GetCell(row)
                Set group = cell.GetCalloutGroup
    
                cellStr = ""
    
                If Not group Is Nothing Then    ' not an empty cell
                    Dim callouts As ObjectSet, it As iterator
    
                    Set callouts = group.GetContents
                    Debug.Assert Not callouts.IsEmpty   ' every group must have at least one callout
    
                    Set it = app.GetClass("it").CreateAObjectIt(callouts)
    
                    it.start
                    Do While it.IsActive
                        valueStr = CalloutValue(it.Current)
    
                        If cellStr = "" Then
                            cellStr = valueStr
                        Else
                            cellStr = cellStr & " " & valueStr
                        End If
                        it.Next
                    Loop
                End If
    
                AddValue rowStr, cellStr
            End If
        Next node
    
        Print #fn, rowStr
    Next row

    Close #fn

    Exit Sub

CantOpenFile:
    MsgBox "Cannot open " + fileName
End Sub

#If False Then  ' diagnostics
Private Function TableColumnType(table As aPartsListTable, column As aTableColumn) As String
    Dim typ As Integer
    typ = table.GetColumnType(column)

    Select Case typ
    Case itemStyleShowId
        TableColumnType = "showID"
    Case itemStyleShowTotalQuantity
        TableColumnType = "showQuantity"
    Case itemStyleShowFileName
        TableColumnType = "showFileName"
    Case itemStyleShowPartNumber
        TableColumnType = "showPartNumber"
    Case itemStyleShowPartDescription
        TableColumnType = "showPartDescription"
    Case itemStyleShowDesigner
        TableColumnType = "showDesigner"
    Case itemStyleShowManager
        TableColumnType = "showManager"
    Case itemStyleShowCompany
        TableColumnType = "showCompany"
    Case itemStyleShowCustom
        TableColumnType = "showCustom(" & table.GetColumnCustomName(column) & ")"
    Case Else
        TableColumnType = "*unknown*"
    End Select
End Function
#End If

Private Function DesignItemValue(itemCallout As aItemCallout) As String
    Dim item As aDesignItem
    Set item = itemCallout.GetDesignItem
    
    Select Case itemCallout.GetStyle
    Case itemStyleShowId
        DesignItemValue = item.GetId
    Case itemStyleShowTotalQuantity
        DesignItemValue = item.GetQuantity
    Case itemStyleShowFileName
        DesignItemValue = item.GetValue(designItemFileName)
    Case itemStyleShowPartNumber
        DesignItemValue = item.GetValue(designItemPartNumber)
    Case itemStyleShowPartDescription
        DesignItemValue = item.GetValue(designItemPartDescription)
    Case itemStyleShowDesigner
        DesignItemValue = item.GetValue(designItemDesigner)
    Case itemStyleShowManager
        DesignItemValue = item.GetValue(designItemManager)
    Case itemStyleShowCompany
        DesignItemValue = item.GetValue(designItemCompany)
    Case itemStyleShowCustom
        DesignItemValue = item.GetCustomValue(itemCallout.GetCustomName)
    Case Else
        DesignItemValue = "*unknown*"
    End Select
End Function

Private Function CalloutValue(callout As aCallout) As String
    If callout.IsA("ItemCallout") Then
        CalloutValue = DesignItemValue(callout)
        If callout.IsUppercase Then
            CalloutValue = UCase(CalloutValue)
        End If
    ElseIf callout.IsA("NoteCallout") Then
        Debug.Print "Note"
        CalloutValue = callout.GetNote.GetText
    ElseIf callout.IsA("DatumCallout") Then
        Debug.Print "Datum"
        CalloutValue = callout.GetDatum.GetName
    ElseIf callout.IsA("DimensionCallout") Then
        Debug.Print "Dimension"
        CalloutValue = callout.GetTolerance.GetDimension.GetValue.GetOutputString
    ElseIf callout.IsA("GeomTolCallout") Then    ' display tolerance value
        Debug.Print "GeomTol"
        Dim dwgValue As zValue
        Set dwgValue = GetApp.CreateValue2(callout.GetTolerance.GetTolerance, 0, 1, 0, 0)
        dwgValue.SetDrawingSpace True
        CalloutValue = dwgValue.GetOutputString
    ElseIf callout.IsA("VariableCallout") Then
        Debug.Print "Variable"
        Dim def As aVariableDef
        Set def = callout.GetDefinition
        
        Select Case callout.GetStyle
        Case 0  ' showName
            CalloutValue = def.GetName
        Case 1  ' showValue
            CalloutValue = def.GetValue.GetOutputString
        Case 2  ' showBoth
            CalloutValue = def.GetName & " = " & def.GetValue.GetOutputString
        End Select
    Else
        Debug.Print "unknown callout type"
        CalloutValue = ""
    End If
End Function

Private Sub AddValue(line As String, value As String)
    If InStr(1, value, sep) > 0 Then  ' enclose in quotes if it contains separator
        value = """" & value & """"
    End If
    If line = "" Then
        line = value
    Else
        line = line & sep & value
    End If
End Sub

